#ifndef __MT_SIMPLE_RESOURCE_H__
#define __MT_SIMPLE_RESOURCE_H__

#include "MTInfoWindow.h"
#include "MTViewer.h"
#include "MTApplication.h"

/***** Type Bitfield Map ******************************************************
 *	
 *	84218421842184218421842184218421
 *	xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
 *	||||||||||||||||||||||||||||||\\-	bytes used
 *	|||||||||||||||||||||||||||||\---	swap bytes
 *	||||||||||||||||||||||||||||\----	filler data
 *	|||||||||||||||||||||||||||\-----	unsigned
 *	||||||||||||||||||||||||||\------	display as decimal
 *	|||||||||||||||||||||||||\-------	store to variable
 *	|||||||||||||||||||||||||				variable idx in data area 1
 *	|||||||||||||||||||||||||				signed adjustment in data area 2
 *	||||||||||||||||||||||||\--------	store constant to variable
 *	||||||||||||||||||||||||				variable idx in data area 1
 *	||||||||||||||||||||||||				constant in data area 2
 *	|||||||||||||||||||||||\---------	start section
 *	|||||||||||||||||||||||					variable idx (# items) in data area 1
 *	|||||||||||||||||||||||					flags in data area 2
 *	|||||||||||||||||||||||						0x01 - repeats to end of file
 *	||||||||||||||||||||||\----------	end section
 *	|||||||||||||||||||||\-----------	data block
 *	|||||||||||||||||||||					number of entries in data area 1
 *	||||||||||||||||||||\------------	special field type
 *	||||||||||||||||||||					type in data area 1
 *	||||||||||||||||||||					flags in data area 2
 *	||||||||||\\\\\\\\\\-------------	data area 1
 *	\\\\\\\\\\-----------------------	data area 2
 *	
 *	0 = end of list
 *	
 *	Special Types:
 *		id	type					data field 2			data field 3 (extend)
 *		0	c string				length
 *		1	float					
 *		2	set to					type					data
 *									0 = absolute			absolute offset
 *									1 = relative			relative offset
 *									2 = variable			variable index
 *		3	jump if					0x1 = equal to			comparison value
 *									0x2 = less than			(info data = jump target)
 *									0x4 = greater than
 *									0x8 = terminate
 *									0x10 = don't advance ptr
 *		4	hex dump										length
 *		5	register operation		0x000 = add				0x0000003F = lhs
 *									0x001 = subtract		0x00000FC0 = rhs
 *									0x002 = multiply		0x0003F000 = tgt
 *									0x003 = divide			
 *									0x004 = skip (lhs)
 *		6	jump abs										(info data = jump target)
 *	
 *****************************************************************************/

#define ONI_RESOURCE_HEADER	{ kTypeUInt32, "ID" }, { kTypeUInt32, "Flags" },

enum
{
	kTypeNoData =					0x00000000,
	kType8Bit =						0x00000001,
	kType16Bit =					0x00000002,
	kType32Bit =					0x00000003,
	kTypeBitSizeMask =				0x00000003,
	
	kTypeSwapBytes =				0x00000004,
	kTypeIsFiller =					0x00000008,
	kTypeIsUnsigned =				0x00000010,
	kTypeDisplayAsDec =				0x00000020,
	kTypeStoreToVariable =			0x00000040,
	kTypeStoreConstantToVariable =	0x00000080,
	kTypeStartSection =				0x00000100,
	kTypeEndSection =				0x00000200,
	kTypeDataBlock =				0x00000400,
	kTypeSpecialType =				0x00000800,
	kTypeEndList =					0x00000000,
	
	kTypeDataArea1Shift =			13,
	kTypeDataArea1ShiftMask =		0x000003FF,
	kTypeDataArea2Shift =			23,
	kTypeDataArea2ShiftMask =		0x000003FF,
	
	kCustomTypeCString =			0,
	kCustomTypeFloat =				1,
	kCustomTypeJumpTo =				2,
	kCustomTypeJumpIf =				3,
	kCustomTypeHexDump =			4,
	kCustomTypeRegisterOp =			5,
	kCustomTypeJumpAbs =			6,
	
	kCustomTypeJumpToAbsolute =		0,
	kCustomTypeJumpToRelative =		1,
	kCustomTypeJumpToVariable =		2,
	
	kCustomTypeJumpIfEqual =		0x01,
	kCustomTypeJumpIfLessThan =		0x02,
	kCustomTypeJumpIfGreaterThan =	0x04,
	kCustomTypeJumpIfNotEqual =		kCustomTypeJumpIfLessThan |
									kCustomTypeJumpIfGreaterThan,
	
	kCustomTypeJumpIfTerminate =	0x08,
	kCustomTypeJumpIfNoAdvance =	0x10,
	
	kCustomTypeROpAdd =				0,
	kCustomTypeROpSubtract =		1,
	kCustomTypeROpMultiply =		2,
	kCustomTypeROpDivide =			3,
	kCustomTypeROpSkip =			4
};

/******************************************************************************
 *	
 *	33222222222211111111110000000000
 *	10987654321098765432109876543210
 *	------------------------xxxxxxxx	type
 *											00000000 = none
 *											00000001 = typeid
 *											00000010 = resource offset
 *											00000011 = resource size
 *	xxxxxxxxxxxxxxxx----------------	data field
 *										[ resource offset ]
 *	----------------------xx--------		file used
 *												00 = 
 *												01 = base file (.dat)
 *												10 = segment file (.sep)
 *												11 = raw file (.raw)
 *	---------------------x----------		is parsed (extend field contains
 *											type, if zero use name)
 *	
 *****************************************************************************/

enum
{
	kTypeInfoTypeMask =				0x000000FF,
	kTypeInfoTypeID =				0x00000001,
	kTypeInfoResourceOffset =		0x00000002,
	kTypeInfoResourceSize =			0x00000003,
	
	kTypeInfoFlagUsesMask =			0x00000003,
	kTypeInfoFlagUsesShift =		8,
	kTypeInfoFlagUsesBaseFile =		0x00000100,
	kTypeInfoFlagUsesSegmentFile =	0x00000200,
	kTypeInfoFlagUsesRawFile =		0x00000300,
	
	kTypeInfoFlagIsParsedMask =		0x00000001,
	kTypeInfoFlagIsParsedShift =	10,
	kTypeInfoFlagIsParsed =			0x00000800,
	
	kTypeInfoTypeID_TypeMask =		0x0000000F,
	kTypeInfoTypeID_Shift =			8,
	kTypeInfoTypeID_None =			0x00000000,
	kTypeInfoTypeID_String =		0x00000100,
	
	kTypeInfoDataMask =				0x0000FFFF,
	kTypeInfoDataShift =			16
};

#define DATA_AREA_1(a)		(((a) & kTypeDataArea1ShiftMask) << kTypeDataArea1Shift)
#define DATA_AREA_2(a)		(((a) & kTypeDataArea2ShiftMask) << kTypeDataArea2Shift)
#define INFO_DATA_AREA(a)	(((a) & kTypeInfoDataMask) << kTypeInfoDataShift)

#define ROP_EXTEND(lhs, rhs, tgt)	(	(((lhs) & 0x3F) <<  0) |	\
										(((rhs) & 0x3F) <<  6) |	\
										(((tgt) & 0x3F) << 12))
enum
{
	kTypeUInt8 =					kType8Bit | kTypeSwapBytes | kTypeIsUnsigned,
	kTypeUInt16 =					kType16Bit | kTypeSwapBytes | kTypeIsUnsigned,
	kTypeUInt32 =					kType32Bit | kTypeSwapBytes | kTypeIsUnsigned,
	
	kTypeSInt8 =					kType8Bit | kTypeSwapBytes,
	kTypeSInt16 =					kType16Bit | kTypeSwapBytes,
	kTypeSInt32 =					kType32Bit | kTypeSwapBytes,
	
	kTypeDecUInt8 =					kTypeUInt8 | kTypeDisplayAsDec,
	kTypeDecUInt16 =				kTypeUInt16 | kTypeDisplayAsDec,
	kTypeDecUInt32 =				kTypeUInt32 | kTypeDisplayAsDec,
	
	kTypeDecSInt8 =					kTypeSInt8 | kTypeDisplayAsDec,
	kTypeDecSInt16 =				kTypeSInt16 | kTypeDisplayAsDec,
	kTypeDecSInt32 =				kTypeSInt32 | kTypeDisplayAsDec,
	
	kTypeFloat =					kTypeSpecialType | DATA_AREA_1(kCustomTypeFloat),
	kTypeCString =					kTypeSpecialType | DATA_AREA_1(kCustomTypeCString)
};

struct MTSimpleResourceEntry
{
	UInt32	data;
	char	title[64];
	UInt32	info;
	SInt32	extend;
};

struct MTSimpleResourceTemplate
{
	MTSimpleResourceEntry	entries[0];
};

struct MTParsedResourceEntry
{
	UInt32		format;
	UInt32		data;
	UInt32		info;
	UInt32		sectionSize;
	SInt32		extend;
	void *		basePtr;
	std::string	title;
	UInt8		divider;
};

class MTSimpleResourceHandler : public MTViewer
{
	public:
				MTSimpleResourceHandler();
		virtual	~MTSimpleResourceHandler();
		
		void	ParseData(UInt8 * basePtr, UInt32 size);
		void	ReparseData(void);
		
		MTParsedResourceEntry *	GetResourceEntry(UInt32 idx)		{ return &resourceEntries[idx]; }
		UInt32					GetResourceEntryFormat(UInt32 idx)	{ return resourceEntries[idx].format; }
		UInt32					GetResourceEntryData(UInt32 idx)	{ return resourceEntries[idx].data; }
		UInt32					GetNumResourceEntries(void)			{ return resourceEntries.size(); }
	
	protected:
		std::vector <MTParsedResourceEntry>	resourceEntries;
		
		virtual void PostProcess(void)	{ }
		virtual void PrivatePostProcess(void) { }
		virtual MTSimpleResourceTemplate *	GetDataTemplate(void) = 0;
		
		UInt8								* localDataCopy;
		UInt32								localDataSize;
	
	private:
		struct LoopStackRecord
		{
			SInt32					loopsRemaining;
			MTSimpleResourceEntry	* loopStart;
			MTParsedResourceEntry	* loopStartEntry;
		};
		
		MTSimpleResourceTemplate			* theTemplate;
};

class MTSimpleResourceInfoWindow : public MTInfoWindow, public MTSimpleResourceHandler
{
	public:
				MTSimpleResourceInfoWindow();
				~MTSimpleResourceInfoWindow();
		
		void	PrivatePostProcess(void);
		
		void	GetWindowName(StringPtr theString);
		
		void	RecieveMessage(UInt32 messageType, UInt32 messageData);
		UInt8	SupportsMessage(UInt32 messageType, UInt32 messageData);
	
	protected:
		UInt8	SupportsSelection(void)	{ return 1; }
		void	HandleListDoubleClick(UInt32 selection);
		
		void	SelectionChanged(void)	{ gTheApp->UpdateMenuState(kChangedMessageSupport); }
		
		void	DoExport(void);
		UInt8	SupportsExport(void)	{ return 1; }
		
		virtual void	DoImport(void);
	
	private:
		void	WriteOutData(void);
		
		void	SpawnEditorWindow(UInt32 idx);
		
		UInt32	exportOffset, exportSize, exportFile;
		UInt8	isDirty;
};

UInt32	ResourceReadData(UInt8 * basePtr, UInt32 format);
UInt32	ResourceReadData(UInt8 * basePtr, UInt32 numBytes, UInt32 swap, UInt32 isUnsigned);
void	ResourceWriteData(UInt8 * basePtr, UInt32 format, UInt32 data);
void	ResourceWriteData(UInt8 * basePtr, UInt32 numBytes, UInt32 swap, UInt32 isUnsigned, UInt32 data);
void	ResourcePrintData(char * buf, MTParsedResourceEntry * entry);
void	ResourcePrintDataSimple(char * buf, UInt32 data, UInt32 format);

extern MTSimpleResourceTemplate	kBINAFormatTemplate;

#endif /* __MT_SIMPLE_RESOURCE_H__ */
